/*
 * Created on 28/06/2005
 */
package com.cav.mserver;

import java.io.*;
import java.net.*;
import java.util.logging.*;

/**
 * A singletone for saving the client statistics.
 * The statistics aggregation methods are either to file, or to UDP server.
 * 
 * @author uri
 */
public class ClientStatistics {

	private static Logger logger = Logger.getLogger(ClientStatistics.class
			.getPackage().getName());

	/** Singletone */
	private static ClientStatistics instance;

	/** The file stream to write to, if "file" is the method (else, it's null) */
	private PrintStream statsStream;

	/** The host to send UDP packet to, if "udp" is the method */
	private InetAddress udpHost;

	/** The port to send UDP packet to, if "udp" is the method */
	private int udpPort;

	/** The UDP socket to use, if "udp" is the method (else, it's null)*/
	private DatagramSocket udpSocket = null;

	/**
	 * Get singletone
	 * @return
	 */
	public static ClientStatistics instance() {
		if (instance == null) {
			synchronized (ClientStatistics.class) {
				instance = new ClientStatistics();
			}
		}
		return instance;
	}

	/**
	 * C'tor. Initializes using configuration.
	 */
	private ClientStatistics() {
		String method = Config.getString("stats.method");
		if (method == null) {
			logger.info("Not keeping statistics");
			return;
		}
		if (method.equals("file")) {
			String statsFileName = Config.getString("stats.file.name");
			try {
				statsStream = new PrintStream(new FileOutputStream(
						statsFileName, true));
			} catch (FileNotFoundException e) {
				logger.log(Level.SEVERE, "Error opening stats file: "
											+ statsFileName, e);
			}
			logger.info("Statistics to file: " + statsFileName);
		} else if (method.equals("udp")) {
			try {
				udpHost = InetAddress.getByName(Config
						.getString("stats.udp.host"));
			} catch (UnknownHostException e) {
				logger.log(
					Level.SEVERE,
					"Error getting host's address: "
		+ Config.getString("stats.udp.host"),
					e);
			}
			try {
				udpPort = Integer.parseInt(Config.getString("stats.udp.port"));
			} catch (NumberFormatException e) {
				logger.log(
					Level.SEVERE,
					"Error getting port number: "
		+ Config.getString("stats.udp.port"),
					e);
			}
			try {
				udpSocket = new DatagramSocket();
			} catch (SocketException e) {
				logger.log(Level.SEVERE, "Error creating statistics socket", e);
			}
			logger.info("Statistics to server (UDP): " + udpHost + "/"
						+ udpPort);
		} else {
			logger.warning("unknown statistics method: " + method);
		}
	}

	/**
	 * Append a statistics record to the file. 
	 * A statistics line is in the following format:<br>
	 * record_time,client_ip,record
	 * <br>
	 * The record time is the time in milliseconds, since 1970 (UTC). If the
	 * first field in the record is a long value, it is assumed to be the gap
	 * between the client saving the record and sending it, in millis.
	 * @param clientAddress
	 * @param record
	 */
	public void appendRecord(InetAddress clientAddress, String record) {
		record = formatRecord(clientAddress, record);
		if (statsStream != null) {
			statsStream.println(record);
		}
		if (udpSocket != null) {
			logger.info("sending: " + record);
			byte[] message = record.getBytes();
			DatagramPacket packet = new DatagramPacket(message, message.length,
					udpHost, udpPort);
			try {
				udpSocket.send(packet);
			} catch (IOException e) {
				logger.log(Level.WARNING, "Error sending statistics packet", e);
			}
		}
	}

	/**
	 * Format a statistics record, in the following format:<br>
	 * record_time,client_ip,record
	 * @param clientAddress
	 * @param record
	 * @return The record, formatted
	 */
	private String formatRecord(InetAddress clientAddress, String record) {
		int commaIndex = record.indexOf(',');
		long recordTime = System.currentTimeMillis();
		if (commaIndex > 0) {
			try {
				recordTime -= Long.parseLong(record.substring(0, commaIndex));
				record = record.substring(commaIndex + 1);
			} catch (NumberFormatException e) {
			} catch (IndexOutOfBoundsException e) {
			}
		}
		record = "" + recordTime + ',' + clientAddress.getHostAddress() + ','
					+ record;
		return record;
	}
}