package com.cav.mserver;

import java.io.*;
import java.text.MessageFormat;
import java.util.*;

import javax.servlet.ServletException;
import javax.servlet.http.*;

/**
 * A Mumps HTTP servlet. It is used to run M code via HTTP protocol.
 * For security reasons, it can only run code that was previously save, with one
 * time key for later execution. The servlet accepts two arguments, passed in 
 * the HTTP GET request:
 * <ul>
 * <li><b>CODE</b> The key of code to run. This is the key by which it was 
 * previously saved.</li>
 * <li><b>ARGS</b> A list of M variables to set prior to executing the M code,
 * passed in a form that can be used for indirected SET command 
 * (e.g. key1=val1,key2=val2)</li>
 * </ul>
 * 
 * @author Uri Schor
 */
public class MumpsServlet extends HttpServlet {

	/** Code to run at the end of a session, to clear the M job */
	private static final String M_SESSION_CLEAR_CODE = "D CLEAR^%ZCAVJSP";

	/** The M routine to execute to run M code previously saved the JSP page */
	private static final MessageFormat REMOTE_EXEC_M_ROUTINE = new MessageFormat(
			"D REMEXEC^%ZCAVJSP(\"{0}\")");

	/**	Empty arguments to pass to the M session */
	private static final Map EMPTY_ARGS = new HashMap();

	/**
	 * Create a new Mumps servlet
	 */
	public MumpsServlet() {
		super();
	}

	/* (non-Javadoc)
	 * @see javax.servlet.http.HttpServlet#doGet(javax.servlet.http.HttpServletRequest, javax.servlet.http.HttpServletResponse)
	 */
	protected void doGet(
		HttpServletRequest request,
		HttpServletResponse response) throws ServletException, IOException {
		response.setContentType("text/xml");
		response.setCharacterEncoding("ISO-8859-1");
		response.setHeader("CACHE-CONTROL", "NO-CACHE");
		Map parameters = new HashMap(request.getParameterMap());
		String[] codeParam = (String[])parameters.remove("CODE");
		if (codeParam == null) {
			response.sendError(1, "No routine name given");
			return;
		}
		String code = codeParam[0];
		String label = REMOTE_EXEC_M_ROUTINE.format(new Object[] { code });
		MumpsSession mSession = null;
		try {
			try {
				mSession = (MumpsSession) MumpsSessionPool.instance()
						.borrowObject();
			}
			catch (Exception e) {
				response.sendError(2, "Error getting Mumps session "
						+ e.toString());
				return;
			}
			OutputStream os = response.getOutputStream();
			os.write("<?xml version=\"1.0\" encoding=\"ISO-8859-1\" ?><root>"
					.getBytes());
			// Clear the session
			clearSession(mSession);
			// Execute command
			InputStreamReader mumpsOutput = new InputStreamReader(mSession
					.execute(label, parameters), "ISO-8859-1");
			int c;
			while ((c = mumpsOutput.read()) != -1) {
				// Write the character while escaping special XML characters
				if (c == '<') {
					os.write("&lt;".getBytes());
				}
				else if (c == '>') {
					os.write("&gt;".getBytes());
				}
				else if (c == '\"') {
					os.write("&quot;".getBytes());
				}
				else if (c == '\'') {
					os.write("&apos;".getBytes());
				}
				else if (c == '&') {
					os.write("&amp;".getBytes());
				}
				else {
					os.write(c);
				}
			}
			os.write("</root>".getBytes());
			// We clear the session here, too,in case it left locks.
			clearSession(mSession);
		}
		finally {
			if (mSession != null) {
				try {
					MumpsSessionPool.instance().returnObject(mSession);
				}
				catch (Exception e) {
					response.sendError(3, "Error releasing Mumps session");
					return;
				}
			}
		}
	}

	/**
	 * Clear a Mumps session.
	 * @param session The session to clear.
	 */
	private void clearSession(MumpsSession mSession) {
		InputStream mumpsOutput = mSession.execute(
			M_SESSION_CLEAR_CODE,
			EMPTY_ARGS);
		try {
			while (mumpsOutput.read() != -1)
				;
		} catch (IOException e) {
			// Nah
		}
	}

}