package com.cav.mserver;

import java.io.*;
import java.net.*;
import java.util.*;
import java.util.logging.*;

/**
 * A TCP server that listens on a socket for client connections and on each 
 * client request executes the Mumps code and return the answer to it.
 * @author Uri Schor
 */
public class MumpsTCPServer {

	private static Logger logger = Logger.getLogger(MumpsTCPServer.class
			.getPackage().getName());

	/** The name of the Mumps code to run within the arguments passed from the client */
	private static final String MUMPS_CODE_ARG = "RTN";

	/** The name of the pseudo argument from the client, that contains statistics */
	private static final String CLIENT_STATS_ARG = "%stats";

	/** The default Mumps code to run */
	private static final String MUMPS_DEFAULT_CODE = "D ^JPOS";

	/** The prompt this server writes to the clients */
	private static final byte[] PROMPT_BYTES = "~~~> \n".getBytes();
	
	/** version number */
	private static final String VERSION = "v1.2";

	/**
	 * A theread that handles a single client
	 * @author Uri Schor
	 */
	private static class ClientThread extends Thread {

		private Socket socket;

		/**
		 * Create a new thread
		 * @param socket
		 */
		public ClientThread(Socket socket) {
			this.socket = socket;
			logger.info("Client connected: " + socket.getInetAddress() + ','
						+ socket.getPort());
		}

		/* (non-Javadoc)
		 * @see java.lang.Thread#run()
		 */
		public void run() {
			try {
				OutputStream os = socket.getOutputStream();
				BufferedOutputStream out = new BufferedOutputStream(os, 1000);
				BufferedReader in = new BufferedReader(new InputStreamReader(
						socket.getInputStream(), "ISO-8859-8"));
				String line = null;
				HashMap params = new HashMap();
				while ((line = in.readLine()) != null) {
					if (line.equalsIgnoreCase("quit")
						|| line.equalsIgnoreCase("q")) {
						break;
					}
					params.clear();
					StringTokenizer tokenizer = new StringTokenizer(line, "&");
					String mcode = MUMPS_DEFAULT_CODE;
					String stats = null;
					while (tokenizer.hasMoreTokens()) {
						String token = tokenizer.nextToken();
						int ind = token.indexOf('=');
						if (ind == -1) {
							continue;
						}
						String var = token.substring(0, ind);
						String value = token.substring(ind + 1);
						if (var.equals(MUMPS_CODE_ARG)) {
							mcode = value;
						} else if (var.equals(CLIENT_STATS_ARG)) {
							stats = value;
						} else {
							params.put(var, "\"" + value + '"');
						}
					}
					MumpsSession mSession = null;
					try {
						mSession = (MumpsSession)MumpsSessionPool.instance()
								.borrowObject();
						InputStream mumpsOutput = mSession.execute(
							mcode,
							params);
						int c;
						while ((c = mumpsOutput.read()) != -1) {
							out.write(c);
							if (c == '\n') {
								// Flush output after each line.
								out.flush();
							}
						}
					} catch (Exception e) {
						logger
								.log(
									Level.SEVERE,
									"Error borrowing Mumps session, or reading from it",
									e);
					} finally {
						try {
							if (mSession != null) {
								MumpsSessionPool.instance().returnObject(
									mSession);
							}
						} catch (Exception e) {
							logger.log(
								Level.SEVERE,
								"Error returning Mumps session ",
								e);
						}
					}
					out.write(PROMPT_BYTES, 0, PROMPT_BYTES.length);
					out.flush();
					if (stats != null) {
						ClientStatistics.instance().appendRecord(
							socket.getInetAddress(),
							stats);
					}
				}
			} catch (IOException e) {
				logger.log(Level.SEVERE, "Error in client thread", e);
			} finally {
				logger.info("Client disconnected: " + socket.getInetAddress()
							+ ',' + socket.getPort());
				try {
					socket.close();
				} catch (IOException e) {
					logger.log(Level.WARNING, "Error closing socket", e);
				}
			}
		}
	}

	/**
	 * Runner
	 * @param args none
	 */
	public static void main(String[] args) {
		LoggingConfiguration.init();

		logger.info("Mumps Proxy Server " + VERSION);
		
		// Create an instance of the session pool, so it's initialized
		MumpsSessionPool.instance();

		ServerSocket serverSocket = null;
		String portStr = Config.getString("server.port");
		int port = 4444;
		if (portStr != null) {
			try {
				port = Integer.parseInt(portStr);
			} catch (NumberFormatException e) {
			}
		}
		try {
			serverSocket = new ServerSocket(port);
		} catch (IOException e) {
			logger.log(Level.SEVERE, "Could not listen on port: " + port, e);
			System.exit(1);
		}

		while (true) {
			Socket clientSocket = null;
			try {
				clientSocket = serverSocket.accept();
				clientSocket.setTcpNoDelay(true);
				new ClientThread(clientSocket).start();
			} catch (IOException e) {
				logger.log(Level.SEVERE, "Accept failed", e);
				System.exit(1);
			}

		}
	}
}